% Convert the simulation results from Katzin et al (2020) Biosystems Engineering to CSV
% Based on:
%   Katzin, D., van Mourik, S., Kempkes, F., & van Henten, E. J. (2020). 
%   GreenLight  An open source model for greenhouses with supplemental lighting: 
%   Evaluation of heat requirements under LED and HPS lamps. 
%   Biosystems Engineering, 194, 6181. https://doi.org/10.1016/j.biosystemseng.2020.03.010
%
% This script takes the simulation results that are contained in a MATLAB
% StateSpaceModel object, and converts them to CSV files that are placed in
% the folder 'CSV Output'.
%
% This script file should be run from its directory, i.e., the active directory in
% MATLAB should be ...\Data from GreenLight (2020)\Simulation data\

% David Katzin, Wageningen University & Research
% david.katzin@wur.nl
% Written in August 2023

%% Set up directories
currentFilename = mfilename();
fullFilePath = mfilename('fullpath');
currentFileDir = strrep(fullFilePath, currentFilename, '');
if isempty(currentFileDir) || contains(currentFileDir, 'Temp\Editor') % This script is run from the console
    currentFileDir = cd(); % the directory from which this script is running
end

addpath(genpath([currentFileDir '\Source code\BramVanthoorModelVer2']));
addpath(genpath([currentFileDir '\Source code\StateSpaceModel']));

% Ensure currentFileDir ends with a  '\'
if currentFileDir(end) ~= '\'
    currentFileDir = [currentFileDir '\'];
end

matlabOutput = [currentFileDir 'MATLAB output\Original data 2019\'];
csvFolder = [currentFileDir 'CSV output\'];

files = dir([matlabOutput '*.mat']);

%% Create CSV files
for iiFile = 1:numel(files)
    fileName = files(iiFile).name;
    
    % Load MATLAB data
    load([matlabOutput fileName], 'bv');
    model = bv;
    
    % Create CSV table
    startTime = datenum(model.t.def);
    time = string(datestr(startTime+(0:300:model.t.val(2))/86400,'dd-mm-yyyy hh:MM'));

    outputTable = ["Time"; time];

    % Outdoor climate 
    tOut = model.d.tOut.val(:,2);
    vpOut = model.d.vpOut.val(:,2);
    co2Out = model.d.co2Out.val(:,2);
    wind = model.d.wind.val(:,2);
    tSky = model.d.tSky.val(:,2);
    tSoil = model.d.tSoOut.val(:,2);
    iGlob = model.d.iGlob.val(:,2);
    firstRow = [...
        "Outdoor temperature (C)" ...
        "Outdoor vapor pressure (Pa)" ...
        "Outdoor CO2 concentration (mg m^{-3})" ...
        "Outdoor wind speed (m s^{-1})" ...
        "Apparent sky temperature (C)" ... 
        "Temperature of external soil layer (C)" ...
        "Global solar radiation (W m^{-2})"];
    outputTable = [outputTable [firstRow; [tOut vpOut co2Out wind tSky tSoil iGlob]]];

    % Indoor climate
    tIn = model.x.tAir.val(:,2);
    vpIn = model.x.vpAir.val(:,2);
    rhIn = model.a.rhIn.val(:,2);
    co2In = model.x.co2Air.val(:,2);
    co2Ppm = model.a.co2InPpm.val(:,2);
    rCan = model.a.rCan.val(:,2);
    firstRow = [...
        "Indoor temperature (C)" ...
        "Indoor vapor pressure (Pa)" ...
        "Indoor relative humidity (%)" ...
        "Indoor CO2 concentration (mg m^{-3})" ...
        "Indoor CO2 concentration (ppm)" ...
        "Global radiation above the canopy (W m^{-2})" ... 
        ];
    outputTable = [outputTable [firstRow; ...
        [tIn vpIn rhIn co2In co2Ppm rCan]]];

    tTop = model.x.tTop.val(:,2);
    vpTop = model.x.vpTop.val(:,2);
    co2Top = model.x.co2Top.val(:,2);
    firstRow = [... 
        "Temperature above the screen (C)" ...
        "Vapor pressure above the screen (Pa)" ...
        "CO2 concentration above the screen (mg m^{-3})" ...
        ];
    outputTable = [outputTable [firstRow; ...
        [tTop vpTop co2Top]]];

    tCovE = model.x.tCovE.val(:,2);
    tCovIn = model.x.tCovIn.val(:,2);
    tThScr = model.x.tThScr.val(:,2);
    tBlScr = model.x.tBlScr.val(:,2);
    tLamp = model.x.tLamp.val(:,2);
    tIntLamp = model.x.tIntLamp.val(:,2);
    tCan = model.x.tCan.val(:,2);
    tCan24 = model.x.tCan24.val(:,2);
    firstRow = [... 
        "External cover temperature (C)" ...
        "Internal cover temperature (C)" ...
        "Thermal screen temperature (C)" ...
        "Blackout screen temperature (C)" ...
        "Lamp temperature (C)" ...
        "Interlamp temperature (C)" ...
        "Canopy temperature (C)" ...
        "Average 24 hour canopy temperature (C)" ...
        ];
    outputTable = [outputTable [firstRow; ...
        [tCovE tCovIn tThScr tBlScr tLamp tIntLamp tCan tCan24]]];

    tPipe = model.x.tPipe.val(:,2);
    tGroPipe = model.x.tGroPipe.val(:,2);
    tFlr = model.x.tFlr.val(:,2);
    tSo1 = model.x.tSo1.val(:,2);
    tSo2 = model.x.tSo2.val(:,2);
    tSo3 = model.x.tSo3.val(:,2);
    tSo4 = model.x.tSo4.val(:,2);
    tSo5 = model.x.tSo5.val(:,2);
    firstRow = [... 
        "Pipe rail temperature (C)" ...
        "Grow pipe temperature (C)" ...
        "Floor temperature (C)" ...
        "Soil layer 1 temperature (C)" ...
        "Soil layer 2 temperature (C)" ...
        "Soil layer 3 temperature (C)" ...
        "Soil layer 4 temperature (C)" ...
        "Soil layer 5 temperature (C)" ...
        ];
    outputTable = [outputTable [firstRow; ...
        [tPipe tGroPipe tFlr tSo1 tSo2 tSo3 tSo4 tSo5]]];

    % Setpoints

    % In the energy use simulation, the setpoint was given as an input
    if isfield(model.d, 'heatSetPoint')
        heatSetPoint = model.d.heatSetPoint.val(:,2);
    else
        heatSetPoint = model.a.heatSetPoint.val(:,2);
    end
    coolSetPoint = model.a.heatMax.val(:,2);
    co2SetPoint = model.a.co2SetPoint.val(:,2);
    rhSetPoint = model.p.rhMax.val*ones(size(co2SetPoint));
    firstRow = [... 
        "Heating setpoint (C)" ...
        "Cooling setpoint (C)" ...
        "CO2 setpoint (ppm)" ...
        "Relative humidity setpoint (%)" ...
        ];
    outputTable = [outputTable [firstRow; ...
        [heatSetPoint coolSetPoint co2SetPoint rhSetPoint]]];

    % Climate control
    boil = model.u.boil.val(:,2);
    groBoil = model.u.groBoil.val(:,2);
    shScr = model.u.shScr.val(:,2);
    thScr = model.u.thScr.val(:,2);
    blScr = model.u.blScr.val(:,2);
    roof = model.u.roof.val(:,2);
    lamp = model.u.lamp.val(:,2);
    intLamp = model.u.intLamp.val(:,2);
    extCo2 = model.u.extCo2.val(:,2);
    firstRow = [... 
        "Pipe rail boiler valve position (0-1)" ...
        "Grow pipe boiler valve position (0-1)" ...
        "Shading screen position (0-1)" ...
        "Thermal screen position (0-1)" ...
        "Blackout screen position (0-1)" ...
        "Roof ventilation position (0-1)" ...
        "Lamp status (0-1)" ...
        "Interlamp status (0-1)" ...
        "CO2 injection valve position (0-1)" ...
        ];
    outputTable = [outputTable [firstRow; ...
        [boil groBoil shScr thScr blScr roof lamp intLamp extCo2]]];

    qRail = model.a.hBoilPipe.val(:,2);
    qGro = model.a.hBoilGroPipe.val(:,2);
    lampIn = model.a.lampIn.val(:,2);
    intLampIn = model.a.intLampIn.val(:,2);
    lampCool = model.a.lampCool.val(:,2);
    vent = model.a.fVentRoof.val(:,2)+model.a.fVentSide.val(:,2);
    co2Inj = model.a.mcExtAir.val(:,2);
    firstRow = [... 
        "Energy supply from the pipe rails (W m^{-2})" ...
        "Energy supply from the grow pipes (W m^{-2})" ...
        "Energy supply from the lamps (W m^{-2})" ...
        "Energy supply from the interlamps (W m^{-2})" ...
        "Lamp cooling (W m^{-2})" ...
        "Ventilation rate (m^{3} m^{-2} s^{-1}" ...
        "CO2 injection rate (mg m^{-2} s^{-1})" ...
        ];
    outputTable = [outputTable [firstRow; ...
        [qRail qGro lampIn intLampIn lampCool vent co2Inj]]];

    % Crop model
    tCanSum = model.x.tCanSum.val(:,2);
    cBuf = model.x.cBuf.val(:,2);
    cLeaf = model.x.cLeaf.val(:,2);
    cStem = model.x.cStem.val(:,2);
    cFruit = model.x.cFruit.val(:,2);
    lai = model.a.lai.val(:,2);
    firstRow = [... 
        "Crop development stage (C day)" ...
        "Carbohydrates in buffer (mg{CH2O} m^{-2})" ...
        "Carbohydrates in leaves (mg{CH2O} m^{-2})" ...
        "Carbohydrates in stems (mg{CH2O} m^{-2})" ...
        "Carbohydrates in fruits (mg{CH2O} m^{-2})" ...
        "Leaf area index (m^2 {leaf} m^{-2} {floor})" ...
        ];
    outputTable = [outputTable [firstRow; ...
        [tCanSum cBuf cLeaf cStem cFruit lai]]];

    mcAirBuf = model.a.mcAirBuf.val(:,2);
    mcBufLeaf = model.a.mcBufLeaf.val(:,2);
    mcBufFruit = model.a.mcBufFruit.val(:,2);
    mcBufStem = model.a.mcBufStem.val(:,2);
    mcBufAir = model.a.mcBufAir.val(:,2);
    mcLeafAir = model.a.mcLeafAir.val(:,2);
    mcFruitAir = model.a.mcFruitHar.val(:,2);
    mcStemAir = model.a.mcStemAir.val(:,2);
    mcLeafHar = model.a.mcLeafHar.val(:,2);
    mcFruitHar = model.a.mcFruitHar.val(:,2);
    mcAirCan = model.a.mcAirCan.val(:,2);
    mvCanAir = model.a.mvCanAir.val(:,2);
    firstRow = [... 
        "Net photosynthesis (mg{CH2O} m^{-2} s^{-1})" ...
        "Carboyhdrate flow from buffer to leaves (mg{CH2O} m^{-2})" ...
        "Carboyhdrate flow from buffer to frtuis (mg{CH2O} m^{-2})" ...
        "Carboyhdrate flow from buffer to stems (mg{CH2O} m^{-2})" ...
        "Growth respiration (mg{CH2O} m^{-2})" ...
        "Leaf maintenance respiration (mg{CH2O} m^{-2})" ...
        "Fruit maintenance respiration (mg{CH2O} m^{-2})" ...
        "Stem maintenance respiration (mg{CH2O} m^{-2})" ...
        "Leaf pruning (mg{CH2O} m^{-2})" ...
        "Fruit harvest (mg{CH2O} m^{-2})" ...
        "Net crop assimilation (mg{CH2O} m^{-2})" ...
        "Canopy transpiration (kg m^{-2} s^{-1})" ...
        ];
    outputTable = [outputTable [firstRow; ...
        [mcAirBuf mcBufLeaf mcBufFruit mcBufStem mcBufAir ...
        mcLeafAir mcFruitAir mcStemAir mcLeafHar mcFruitHar ...
        mcAirCan mvCanAir]]];
    
    % Write file - notice this only works from MATLAB R2019a onwards
    writematrix(outputTable, [csvFolder fileName(1:end-4) '.csv']);
end